////----------------------------------------------------------------------------------
// File:        CVideoFile.h
// SDK Version: 1.0.2
//
// SPDX-FileCopyrightText: Copyright (c) 2023 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
// SPDX-License-Identifier: LicenseRef-NvidiaProprietary
//
// NVIDIA CORPORATION, its affiliates and licensors retain all intellectual
// property and proprietary rights in and to this material, related
// documentation and any modifications thereto. Any use, reproduction,
// disclosure or distribution of this material and related documentation
// without an express license agreement from NVIDIA CORPORATION or
// its affiliates is strictly prohibited.
//
//----------------------------------------------------------------------------------

///////////////////////////////////////////
// CVideoFile.h
//
// Read uncompressed formats from file into a buffer
//

#pragma once

#include <stdio.h>

// FourCC and format types:
#include "nvFourCC.h"

class CVideoFile
{

public:

    CVideoFile(bool bUseTexturePitch = false);
    virtual ~CVideoFile();

    // Opens/closes the specified video file.
    HRESULT Open(char *strFilePath);
    void    Close();
    // Copies the video data from the file to the destination buffer.
    HRESULT FillBuffer(char* pDstBuf, int nDstFourCC, int nDstPitch, bool bDoubleSize = false);

    // Increments/decrements the video frame to read.
    HRESULT IncrementFrame();
    HRESULT DecrementFrame();
    int IncrementFrameCyclic();
    int DecrementFrameCyclic();

    // Accessor methods for getting the properties of the video.
    DWORD FourCC()          const { return m_nFourCC; }
    int Width()             const { return m_nWidth; }
    int Height()            const { return m_nHeight; }
    int Pitch()             const { return m_bUseTexturePitch ? m_nTexturePitch : m_nPitch; }
    int BPP()               const { return m_nBPP; }
    int NumFrames()         const { return m_nNumFrames; }

    char *ClipName()              { return m_strClipName; }
    char *FourCCString()          { return m_strFourCC; }


private:

    // Helper function for actual reading of video data from file.
    inline void Read(char* const pDstBuf, long lOffset, int nAmount);

    FILE*       m_File;                 // file stream object
    char        m_strFourCC[6];         // FourCC string (ex. NV12)
    DWORD       m_nFourCC;              // FourCC (ex. FOURCC_NV12)
    int         m_nWidth;               // video frame width
    int         m_nHeight;              // video frame height
    int         m_nPitch;               // video frame pitch
    int         m_nTexturePitch;        // video frame texture pitch
    bool        m_bUseTexturePitch;
    int         m_nBPP;                 // bytes per pixel
    int         m_nNumFrames;           // number of frames in the file
    char        m_strClipName[256];     // name of the clip (ex. Yozakura)

    long        m_lCurrentOffet;
    int         m_nCurrentFrameIndex;   // current frame index (starting from 0)
};
