////----------------------------------------------------------------------------------
// File:        rtx_video_api.h
// SDK Version: 1.0.2
//
// SPDX-FileCopyrightText: Copyright (c) 2024 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
// SPDX-License-Identifier: LicenseRef-NvidiaProprietary
//
// NVIDIA CORPORATION, its affiliates and licensors retain all intellectual
// property and proprietary rights in and to this material, related
// documentation and any modifications thereto. Any use, reproduction,
// disclosure or distribution of this material and related documentation
// without an express license agreement from NVIDIA CORPORATION or
// its affiliates is strictly prohibited.
//
//----------------------------------------------------------------------------------


///////////////////////////////////////////
// rtx_video_api.h
//
// This provides a simple interface to use the RTX Video SDK.
// There are create, evaluate, shutdown functions for DX11, DX12.
// The application calls create at startup, shutdown at exit.
// To apply the VSR and/or TrueHDR, call evalulate with input and output textures, and sizes.
// The input textures must be ARGB.
// If TrueHDR is enabled, the output texture must be A2BGR10. Otherwise ARGB.
//

#pragma once

#if defined(_WIN32)
#include <cstdint>
#else
typedef unsigned int uint32_t;
typedef unsigned long uint64_t;
#endif

#ifdef __cplusplus
#define C_API "C"
#else
#define C_API
#endif

typedef struct
{
    uint32_t QualityLevel;
} API_VSR_Setting;

typedef struct
{
    uint32_t Contrast;
    uint32_t Saturation;
    uint32_t MiddleGray;
    uint32_t MaxLuminance;
} API_THDR_Setting;

typedef unsigned int API_BOOL;
#define API_BOOL_FAIL       0
#define API_BOOL_SUCCESS    1

typedef struct
{
    uint32_t left;
    uint32_t top;
    uint32_t right;
    uint32_t bottom;
} API_RECT;

////////////////////////////////////////////////////////////////////
// DX11
struct ID3D11Device;
struct ID3D11Texture2D;

// For DX11, first create ID3D11Device and pass it in.
extern C_API API_BOOL rtx_video_api_dx11_create(ID3D11Device* pD3DDevice, API_BOOL THDREnable, API_BOOL VSREnable);
// Ideally the output surface should be created with D3D11_BIND_UNORDERED_ACCESS, but is not required.
extern C_API API_BOOL rtx_video_api_dx11_evaluate(ID3D11Texture2D* pInput, ID3D11Texture2D* pOutput, API_RECT inputRect, API_RECT outputRect, API_VSR_Setting* pVSRSetting, API_THDR_Setting* pTHDRSetting);
extern C_API void rtx_video_api_dx11_shutdown();

////////////////////////////////////////////////////////////////////
// DX12
struct ID3D12Device;
struct ID3D12Resource;
struct ID3D12Fence;

// For DX12, first create ID3D12Device and pass it in, along with GPU selection.
extern C_API API_BOOL rtx_video_api_dx12_create(ID3D12Device* pD3DDevice, uint32_t uGPUNodeMask, uint32_t uGPUVisibleNodeMask, API_BOOL THDREnable, API_BOOL VSREnable);
// Ideally the output surface should be created with D3D12_RESOURCE_FLAG_ALLOW_UNORDERED_ACCESS, but is not required. Fences will be waited by GPU on and signaled, incrementing value.
extern C_API API_BOOL rtx_video_api_dx12_evaluate(ID3D12Resource* pInput, ID3D12Resource* pOutput, ID3D12Fence* pInFence, uint64_t& qwInFenceValue, ID3D12Fence* pOutFence, uint64_t& qwOutFenceValue,
                                                  API_RECT inputRect, API_RECT outputRect, API_VSR_Setting* pVSRSetting, API_THDR_Setting* pTHDRSetting);
extern C_API void rtx_video_api_dx12_shutdown();
